package org.apache.torque.generator.control;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.io.FileUtils;
import org.apache.torque.generator.BaseTest;
import org.apache.torque.generator.configuration.UnitDescriptor;
import org.apache.torque.generator.configuration.paths.CustomProjectPaths;
import org.apache.torque.generator.configuration.paths.DefaultTorqueGeneratorPaths;
import org.apache.torque.generator.configuration.paths.Maven2DirectoryProjectPaths;
import org.junit.Test;

public class PropertyToJavaGenerationTest extends BaseTest
{
    /**
     * Tests that the property to java generation works.
     *
     * @throws Exception if the test fails.
     */
    @Test
    public void testPropertyToJavaGeneration() throws Exception
    {
        // prepare
        File targetDir = new File("target/test/propertyToJava");
        FileUtils.deleteDirectory(targetDir);
        Controller controller = new Controller();
        List<UnitDescriptor> unitDescriptors = new ArrayList<UnitDescriptor>();
        CustomProjectPaths projectPaths = new CustomProjectPaths(
                new Maven2DirectoryProjectPaths(
                        new File("src/test/propertyToJava")));
        projectPaths.setOutputDirectory(null, targetDir);
        unitDescriptors.add(new UnitDescriptor(
                UnitDescriptor.Packaging.DIRECTORY,
                projectPaths,
                new DefaultTorqueGeneratorPaths()));

        // execute
        controller.run(unitDescriptors);

        // verify
        assertTrue(targetDir.exists());
        File logoResultFile = new File(targetDir, "logo.png");
        assertTrue(logoResultFile.exists());
        byte[] expectedContent = FileUtils.readFileToByteArray(
            new File("src/test/propertyToJava/src/main/torque-gen/resources/torque-logo-new.png"));
        assertArrayEquals(
                expectedContent,
                FileUtils.readFileToByteArray(logoResultFile));
        File propertiesResultFile
                = new File(targetDir, "Properties.properties");
        assertTrue(propertiesResultFile.exists());
        File propertiesExpectedFile
            = new File("src/test/propertyToJava/expectedProperties.properties");
        assertEquals(FileUtils.readFileToString(propertiesExpectedFile),
                FileUtils.readFileToString(propertiesResultFile));
    }

    /**
     * Tests that the property to java generation works with debug output
     * switched on.
     *
     * @throws Exception if the test fails.
     */
    @Test
    public void testPropertyToJavaGenerationDebugOutput() throws Exception
    {
        // prepare
        File targetDir = new File("target/test/propertyToJava");
        FileUtils.deleteDirectory(targetDir);
        Controller controller = new Controller();
        List<UnitDescriptor> unitDescriptors = new ArrayList<UnitDescriptor>();
        CustomProjectPaths projectPaths = new CustomProjectPaths(
                new Maven2DirectoryProjectPaths(
                        new File("src/test/propertyToJava")));
        projectPaths.setOutputDirectory(null, targetDir);
        UnitDescriptor unitDescriptor = new UnitDescriptor(
                UnitDescriptor.Packaging.DIRECTORY,
                projectPaths,
                new DefaultTorqueGeneratorPaths());
        unitDescriptor.setAddDebuggingInfoToOutput(true);
        unitDescriptors.add(unitDescriptor);

        // execute
        controller.run(unitDescriptors);

        // verify
        assertTrue(targetDir.exists());
        File logoResultFile = new File(targetDir, "logo.png");
        assertTrue(logoResultFile.exists());
        byte[] expectedContent = FileUtils.readFileToByteArray(
                new File("src/test/propertyToJava/src/main/torque-gen/resources/torque-logo-new.png"));
        assertArrayEquals(
                expectedContent,
                FileUtils.readFileToByteArray(logoResultFile));
        File propertiesResultFile
                = new File(targetDir, "Properties.properties");
        assertTrue(propertiesResultFile.exists());
        File propertiesExpectedFile
                = new File("src/test/propertyToJava/expectedPropertiesDebugOutput.properties");
        assertEquals(FileUtils.readFileToString(propertiesExpectedFile),
                FileUtils.readFileToString(propertiesResultFile));
    }
}
