// Copyright The OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package jaegerreceiver

import (
	"go.opentelemetry.io/collector/config/configgrpc"
	"go.opentelemetry.io/collector/config/configmodels"
	"go.opentelemetry.io/collector/config/configtls"
)

// The config field name to load the protocol map from
const protocolsFieldName = "protocols"

// RemoteSamplingConfig defines config key for remote sampling fetch endpoint
type RemoteSamplingConfig struct {
	HostEndpoint                  string `mapstructure:"host_endpoint"`
	StrategyFile                  string `mapstructure:"strategy_file"`
	configgrpc.GRPCClientSettings `mapstructure:",squash"`
}

type SecureSetting struct {
	configmodels.ReceiverSettings `mapstructure:",squash"` // squash ensures fields are correctly decoded in embedded struct
	// Configures the receiver to use TLS.
	// The default value is nil, which will cause the receiver to not use TLS.
	TLSCredentials *configtls.TLSSetting `mapstructure:"tls_credentials, omitempty"`
}

// Config defines configuration for Jaeger receiver.
type Config struct {
	TypeVal        configmodels.Type         `mapstructure:"-"`
	NameVal        string                    `mapstructure:"-"`
	Protocols      map[string]*SecureSetting `mapstructure:"protocols"`
	RemoteSampling *RemoteSamplingConfig     `mapstructure:"remote_sampling"`
}

// Name gets the receiver name.
func (rs *Config) Name() string {
	return rs.NameVal
}

// SetName sets the receiver name.
func (rs *Config) SetName(name string) {
	rs.NameVal = name
}

// Type sets the receiver type.
func (rs *Config) Type() configmodels.Type {
	return rs.TypeVal
}

// SetType sets the receiver type.
func (rs *Config) SetType(typeStr configmodels.Type) {
	rs.TypeVal = typeStr
}
